/*=======================================================*/
/*
 *  Listing 1.  showTimer.c 
 *
 *  This program shows how a device is used by opening
 *  and using the timer device, then closing it.
 *
 *  This program requires V36 or higher.
*/
/*=======================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <exec/types.h>
#include <exec/memory.h>
#include <devices/timer.h>
#include <clib/exec_protos.h>
#include <clib/dos_protos.h>
#include <clib/alib_protos.h>

/*
 *  To use the timer device arithmetic functions, we need 
 *  a library base, a timer request, and a timeval.  This
 *  program does not use the library-type functions 
 *  implemented by the timer device.
*/

struct Library *TimerBase = NULL;
struct timerequest *timerIO = NULL;
struct timeval *timeVal = NULL;

void main()
{
   struct MsgPort *msgPort = NULL;
   unsigned long reqLen = sizeof(struct timerequest);
   struct timerequest *ioReqCopy = NULL;
   int    copySent = 0;
   BYTE   err;

   /*
    *  create a message port.  Devices require the
    *  address of a message port to which they send
    *  replies to commands
   */

   msgPort = CreateMsgPort();
   if ( ! msgPort )
   {
      printf("Can't create message port\n");
      exit(1);
   }

   /*
    *  create io request.  Each device can define its own
    *  device-specific io request by extending either
    *  the standard IORequest structure or by extending
    *  the IOStdRequest structure (defined in exec/io.h).
    *  The CreateIORequest() call creates and initializes
    *  an io request.
   */

   timerIO = (struct timerequest*)
                  CreateIORequest(msgPort,reqLen);
   if ( ! timerIO )
   {
      printf("Can't create IO request\n");
      goto allDone;
   }

   /*
    *  open the device.  The timer has five different 
    *  units.  We will use the vertical blank interval
    *  timer unit.  The defines for the units are in
    *  devices/timer.h.  The device name, TIMERNAME,
    *  is also defined in timer.h.
    *
    *  If successful, the call to OpenDevice() fills in
    *  the rest of the fields that must be initialized
    *  for specific devices. 
   */

   printf("\nCalling OpenDevice for device %s\n",
          TIMERNAME);

   err = OpenDevice(TIMERNAME,UNIT_VBLANK,
                    (struct IORequest*)timerIO,0);
   if ( err )
   {
      printf("Can't open %s\n",TIMERNAME);
      goto allDone;
   }
   else
      printf("%s successfully opened",TIMERNAME);
  

   /*
    *  make a copy of the initialized io request to
    *  use in sending multiple asynchronous commands
    *  to the timer.
   */

   ioReqCopy = (struct timerequest*)
               AllocVec(sizeof(struct timerequest),
                        MEMF_PUBLIC|MEMF_CLEAR);
   if ( ! ioReqCopy )
      printf("Can't allocate memory\n");
   else
      CopyMem((APTR)timerIO,(APTR)ioReqCopy,
              sizeof(struct timerequest));
   
   /*
    *  set up the library base - same as the address of 
    *  the device.  This must be done if the timer
    *  'time arithmetic' functions are used.
   */

   TimerBase = (struct Library*)
                           timerIO->tr_node.io_Device;

   /*
    *  Set up the io request and send a command to the
    *  timer using DoIO().  DoIO() is synchronous:  it
    *  does not return control to a program until a
    *  device completes the command.
    *
    *  Calling DoIO() is the simplest way to access
    *  a device command.
   */

   printf("\nCalling DoIO() with time delay request\n");

   timerIO->tr_node.io_Command = TR_ADDREQUEST;
   timerIO->tr_time.tv_secs  = 5;
   timerIO->tr_time.tv_micro = 0;

   err = DoIO((struct IORequest*)timerIO);
   if ( err )
   {
      printf("Error from DoIO, err = %d\n",
             timerIO->tr_node.io_Error);
      goto allDone;
   }

   printf("DoIO returned without error\n");

   /*
    *  set up io request and send asynchronous
    *  commands to the timer by calling BeginIO() and
    *  SendIO().  These procedures return without
    *  waiting for a device to complete a command.
    *  The completion status of asynchronous commands
    *  is checked by calling CheckIO().  The device
    *  lets a program know when the command is finished
    *  by signaling the program's message port.  Thus
    *  the io requests must be read back from the port.
    *  The simplest way to do this is by calling WaitIO(),
    *  which takes care of all the message removal
    *  details.
    *  
    *  Request "quick io" for the call to BeginIO to show
    *  how this bit is used, even though the delay request
    *  can not be completed immediately.
   */

   printf("\nCalling BeginIO with time delay request\n");

   timerIO->tr_time.tv_secs  = 5;
   timerIO->tr_time.tv_micro = 0;
   timerIO->tr_node.io_Flags &= IOF_QUICK;

   BeginIO((struct IORequest*)timerIO);
   if ( timerIO->tr_node.io_Error )
   {
      printf("Error from BeginIO, err = %d\n",
             timerIO->tr_node.io_Error);
      goto allDone;
   }

   if( ioReqCopy )
   {
      printf("\nCalling SendIO with time delay request\n");

      ioReqCopy->tr_node.io_Command = TR_ADDREQUEST;
      ioReqCopy->tr_time.tv_secs  = 60;
      ioReqCopy->tr_time.tv_micro = 0;

      SendIO((struct IORequest*)ioReqCopy);
      if ( ioReqCopy->tr_node.io_Error )
         printf("Error from SendIO, err = %d\n",
               ioReqCopy->tr_node.io_Error);
      copySent = 1;
   }

   if ( ! (timerIO->tr_node.io_Flags & IOF_QUICK) )
   {
      printf("Waiting for BeginIO to complete...\n");
      WaitIO((struct IORequest*)timerIO);
   }

   printf("BeginIO() time delay complete\n");

   /*
    *  now abort the io request.  All device io
    *  requests must be completed or aborted before
    *  a program ends.  Even after calling AbortIO(),
    *  the request must be returned and removed from
    *  the message port!
   */

   if ( ioReqCopy && copySent )
   {
      if ( ! CheckIO((struct IORequest*)ioReqCopy) )
      {
         printf("Aborting SendIO command...\n");
         AbortIO((struct IORequest*)ioReqCopy);
      }
      else
         printf(
            "SendIO time delay complete...\n");
         
      printf("Waiting for io completion message...\n");

      WaitIO((struct IORequest*)ioReqCopy);
   }

allDone:

   /*
    *  clean up  
    *  we do not need to call CloseLibrary() even
    *  though we set up a library base (TimerBase).
    *  Closing the device takes care of this. 
   */

   if ( ioReqCopy )
      FreeVec( ioReqCopy );

   if ( timerIO )
   {
      /*
       *  Make sure we got a timerIO and also make
       *  sure the device was successfully opened.
      */

      if ( timerIO->tr_node.io_Device )
      {   
         printf("\nCalling CloseDevice\n");
         CloseDevice((struct IORequest*)timerIO);
      }

      printf("Deleting timer io request\n");
      DeleteIORequest(timerIO);
   }

   if( msgPort )
   {
      printf("Deleting message port\n\n");
      DeleteMsgPort(msgPort);      
   }

   exit(0);
}

